#if NUNIT
#region Directives
//-----------------------***
using System;
using System.Collections;
using System.Data;
using System.Text;
using NUnit.Framework;
using gov.va.med.vbecs.UnitTests;

using TABLES = gov.va.med.vbecs.Common.VbecsTables;
using SPROCS = gov.va.med.vbecs.Common.VbecsStoredProcs;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;

using BOL = gov.va.med.vbecs.BOL;
using COMMON = gov.va.med.vbecs.Common;
using DAL = gov.va.med.vbecs.DAL;
using VAL = gov.va.med.vbecs.VAL;
using gov.va.med.vbecs.VAL.UnitTests;
//-----------------------***
using System.Data.SqlClient;

#endregion

namespace gov.va.med.vbecs.UnitTests
{

	/// <summary>
	/// Utility methods used across VBECS unit tests.
	/// </summary>
	public class UnitTestUtil
	{
		public const string Alphabet = "ABCDEFGHIJKLMNOPQRSTUVWXYZ";

		private UnitTestUtil(){}

		private static void VerifySqlText( string sqlText )
		{
			if( sqlText == null ) 
				Assert.Fail( "sqlText argument supplied to unit test utility method was null." );
		}

		public static DataTable RunSQLGetDataTable( string sqlText )
		{
			return RunSQLGetDataTable( sqlText, false );
		}

		public static DataTable RunSQLGetDataTable( string sqlText, bool throwOnNoRows )
		{
			VerifySqlText( sqlText );

			using( SqlConnection _cn = new SqlConnection( COMMON.VbecsConfig.Current.PrimaryDbConnectionString ) )
			{
				SqlCommand _cmd = _cn.CreateCommand();
				_cmd.CommandText = sqlText;
				
				SqlDataAdapter _da = new SqlDataAdapter( _cmd );
				DataTable _dt = new DataTable();

				_cn.Open();
				_da.Fill( _dt );

				// May be it will be more suitable to call Assertion.Fail instead of throwing BusinessObjectException? 
				// The latter may lead to confusion. (Stas)
				// 
				if( throwOnNoRows && _dt.Rows.Count == 0 )
					throw new BOL.BusinessObjectException( "No unit test data for >>" + sqlText + "<<" );

				return _dt;
			}
		}

		public static DataRow RunSQLGetDataRow( string sqlText )
		{
			VerifySqlText( sqlText );

			DataTable _dt = RunSQLGetDataTable( sqlText, false );

			if( _dt.Rows.Count != 1 )
				Assert.Fail( String.Format( "{0} rows was fetched in RunSQLGetDataRow instead of 1 row expected.", _dt.Rows.Count ) );

			return _dt.Rows[0];
		}

		public static DataRow RunSQLGetDataRowDontThrowExceptionIfNoRowIsReturned( string sqlText )
		{
			VerifySqlText( sqlText );

			DataTable dt = RunSQLGetDataTable( sqlText, false );

			return dt.Rows != null && dt.Rows.Count > 0 ? dt.Rows[0] : null;
		}

		public static int RunSQL( string sqlText )
		{
			VerifySqlText( sqlText );

			int retValue = 0;

			using( SqlConnection _cn = new SqlConnection( COMMON.VbecsConfig.Current.PrimaryDbConnectionString ) )
			{
				SqlCommand _cmd = _cn.CreateCommand();
				_cmd.CommandText = sqlText;

				_cn.Open();
				retValue = _cmd.ExecuteNonQuery();
			}

			return(retValue);
		}

		/// <summary>
		/// Runs the input parameter sql statements in a transaction
		/// </summary>
		/// <param name="sqlStatements"></param>
		/// <returns></returns>
		public static int RunSQLInTransaction(params string[] sqlStatements)
		{
			int retValue = 0;

			using( SqlConnection _cn = new SqlConnection( COMMON.VbecsConfig.Current.PrimaryDbConnectionString ) )
			{
				SqlTransaction tran = _cn.BeginTransaction();
				_cn.Open();

				foreach(string tmpSQL in sqlStatements)
				{
					VerifySqlText( tmpSQL );

					SqlCommand _cmd = new SqlCommand(tmpSQL, _cn, tran);

					retValue = _cmd.ExecuteNonQuery();
				}

				tran.Commit();
			}

			//Not really sure if I should do anything with return value, since it could be any number really...but returning it anyway
			return(retValue);

		}


		public static string GetRandomString( int maxChars )
		{
			if( maxChars < 1 )
				throw( new ArgumentOutOfRangeException( "maxChars" ) );

			Random _rnd = new Random( unchecked( (int)DateTime.Now.Ticks ) ); 
			
			int _strLen = _rnd.Next( maxChars );
			if( _strLen == 0 )
				_strLen = 1;

			StringBuilder _sb = new StringBuilder( _strLen );

			for( int i = 0; i < _strLen; i++ )
				_sb.Append( Alphabet[ _rnd.Next( Alphabet.Length - 1 ) ] );

			return _sb.ToString();
		}

		public static string GetRandomUniqueUserDuz()
		{
			Random _rnd = new Random( unchecked( (int)DateTime.Now.Ticks ) ); 

			for( int _safetyCounter = 0; _safetyCounter < 1000; _safetyCounter++ )
			{
				string _duz = _rnd.Next(1, 99999).ToString();

				DataTable _dt = UnitTestUtil.RunSQLGetDataTable( "select top 1 1 from VbecsUser where UserDuz = '" + _duz + "'", false );

				if( _dt.Rows.Count == 0 )
					return _duz;
			}

			Assert.Fail( "Unable to create unique VbecsUser DUZ!" );
			throw( new Exception( "Compile stub" ) );
		}
	}
}
#endif